/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.core;

import org.apiguardian.api.API;
import org.opentest4j.reporting.events.api.ChildElement;
import org.opentest4j.reporting.events.api.Context;
import org.opentest4j.reporting.schema.Namespace;
import org.opentest4j.reporting.schema.QualifiedName;

import java.time.LocalDateTime;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * The {@code output} element of the core namespace.
 *
 * @since 0.2.0
 */
@API(status = EXPERIMENTAL, since = "0.2.0")
public class Output extends ChildElement<Attachments, Output> {

	/**
	 * Qualified name of the {@code output} element.
	 */
	public static final QualifiedName ELEMENT = QualifiedName.of(Namespace.REPORTING_CORE, "output");

	/**
	 * Qualified name of the {@code time} attribute.
	 */
	public static final QualifiedName TIME = QualifiedName.of(Namespace.REPORTING_CORE, "time");

	/**
	 * Qualified name of the {@code source} attribute.
	 */
	public static final QualifiedName SOURCE = QualifiedName.of(Namespace.REPORTING_CORE, "source");

	Output(Context context) {
		super(context, ELEMENT);
	}

	/**
	 * Set the {@code time} attribute of this element.
	 *
	 * @param timestamp the timestamp to set
	 * @return this element
	 */
	public Output withTime(LocalDateTime timestamp) {
		withAttribute(TIME, timestamp.toString());
		return this;
	}

	/**
	 * Set the {@code source} attribute of this element.
	 *
	 * @param source the source to set
	 * @return this element
	 */
	public Output withSource(String source) {
		withAttribute(SOURCE, source);
		return this;
	}

	/**
	 * Set the content of this element.
	 *
	 * @param content the content to set
	 * @return this element
	 */
	public Output withContent(String content) {
		withCDataSection(content);
		return this;
	}
}
