// @HEADER
// *****************************************************************************
//        MueLu: A package for multigrid based preconditioning
//
// Copyright 2012 NTESS and the MueLu contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

#ifndef MUELU_CLASSICALMAPFACTORY_DECL_HPP_
#define MUELU_CLASSICALMAPFACTORY_DECL_HPP_

#include "Xpetra_StridedMapFactory_fwd.hpp"
#include "Xpetra_Import_fwd.hpp"
#include "Xpetra_Vector_fwd.hpp"
#include "Xpetra_VectorFactory_fwd.hpp"

#include "MueLu_ConfigDefs.hpp"
#include "MueLu_SingleLevelFactoryBase.hpp"
#include "MueLu_ClassicalMapFactory_fwd.hpp"
#include "MueLu_LWGraph_fwd.hpp"
#include "MueLu_Level_fwd.hpp"
#include "MueLu_Exceptions.hpp"

#include "MueLu_LWGraph_fwd.hpp"
#include "MueLu_LWGraph_kokkos_fwd.hpp"

namespace MueLu {

/*!
  @class ClassicalMapFactory class.
  @brief Factory for generating F/C-splitting and a coarse level map. Used by ClassicalPFactory.

  @ingroup MueLuTransferClasses

  ## Input/output ##

  ### User parameters of this factory ###
  Parameter | type | default | master.xml | validated | requested | description
  ----------|------|---------|:----------:|:---------:|:---------:|------------
  Graph | Factory | null | | * | * | Generating factory for graph.
  The * in the @c master.xml column denotes that the parameter is defined in the @c master.xml file.<br>
  The * in the @c validated column means that the parameter is declared in the list of valid input parameters (see @c GetValidParameters() ).<br>
  The * in the @c requested column states that the data is requested as input with all dependencies (see @c DeclareInput() ).


  ### Variables provided by this factory ###

  After @c Build() the following data is available (if requested)

  Parameter | generated by | description
  ----------|--------------|------------
  | Colors | ClassicalMapFactory | ArrayRCP<LO> of colors
  | CoarseMap | CoarseMapFactory | Map containing the coarse map used as domain map in the classical prolongator

*/

template <class Scalar        = DefaultScalar,
          class LocalOrdinal  = DefaultLocalOrdinal,
          class GlobalOrdinal = DefaultGlobalOrdinal,
          class Node          = DefaultNode>
class ClassicalMapFactory : public SingleLevelFactoryBase {
#undef MUELU_CLASSICALMAPFACTORY_SHORT
#include "MueLu_UseShortNames.hpp"

 public:
  //! F/C/Dirichlet point type
  typedef enum { F_PT         = -1,
                 UNASSIGNED   = 0,
                 C_PT         = 1,
                 DIRICHLET_PT = 2 } point_type;

  //! @name Input
  //@{

  RCP<const ParameterList> GetValidParameterList() const override;

  /*!
    @brief Specifies the data that this class needs, and the factories that generate that data.

    If the Build method of this class requires some data, but the generating factory is not specified in DeclareInput,
    then this class will fall back to the settings in FactoryManager.
  */
  void DeclareInput(Level &currentLevel) const override;

  //@}

  //! @name Build methods.
  //@{

  //! Build an object with this factory.
  void Build(Level &currentLevel) const override;

  //@}

 protected:
  virtual void GenerateCoarseMap(const Map &fineMap, LO num_c_points, Teuchos::RCP<const Map> &coarseMap) const;

  virtual void DoGraphColoring(const LWGraph &graph, Teuchos::ArrayRCP<LO> &myColors, LO &numColors) const;

  virtual void DoMISNaive(const LWGraph &graph, Teuchos::ArrayRCP<LO> &myColors, LO &numColors) const;

  virtual void DoDistributedGraphColoring(RCP<const LWGraph> &graph, Teuchos::ArrayRCP<LO> &myColors, LO &numColors) const;

};  // class ClassicalMapFactory

}  // namespace MueLu

#define MUELU_CLASSICALMAPFACTORY_SHORT
#endif /* MUELU_CLASSICALMAPFACTORY_DECL_HPP_ */
